package com.agilex.healthcare.directscheduling.datalayer;

import javax.annotation.Resource;

import com.agilex.healthcare.directscheduling.domain.AppointmentSlotFilter;
import com.agilex.healthcare.directscheduling.domain.AppointmentTimeSlots;
import com.agilex.healthcare.directscheduling.domain.BookAppointment;
import com.agilex.healthcare.directscheduling.domain.BookedAppointments;
import com.agilex.healthcare.directscheduling.domain.CancelAppointment;
import com.agilex.healthcare.directscheduling.domain.CancelReasons;
import com.agilex.healthcare.directscheduling.mdws.connection.MdwsConfiguration;
import com.agilex.healthcare.directscheduling.mdws.connection.SchedulingConnection;
import com.agilex.healthcare.directscheduling.mdws.generatedwsdl.schedulingservice.*;
import com.agilex.healthcare.directscheduling.mdws.translator.*;
import com.agilex.healthcare.directscheduling.utils.DateHelper;
import com.agilex.healthcare.directscheduling.validator.MdwsResponseValidator;
import com.agilex.healthcare.directscheduling.validator.VARResponseValidator;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.utility.NullChecker;
import gov.va.vamf.scheduling.direct.domain.ClinicProfile;

import java.util.Date;

public class SchedulingDataLayerMdws extends MdwsSchedulingDataLayer implements SchedulingDataLayer {

    @Resource
    VARResponseValidator varresponseValidator;

    private final static org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(SchedulingDataLayerMdws.class);

    public SchedulingDataLayerMdws(MdwsConfiguration mdwsConfiguration) {
        super(mdwsConfiguration);
    }

    public BookAppointment bookPatientAppointment(BookAppointment bookAppointment, String siteCode) {
        logger.debug("Book patient appointment");
        String vistaDate = DateHelper.convertDateTimeToVistaDateTimeFormat(bookAppointment.getDateTime());
        String vistaDesiredDate = removeTrailingZeros(DateHelper.convertDateTimeToVistaDateTimeFormat(bookAppointment.getDesiredDate()));

        SchedulingConnection authenticatedConnection = super.getAuthenticatedConnection(siteCode);
        AppointmentTO lvAppointmentTO = authenticatedConnection.getPort().makeVeteranAppointment(siteCode,
            bookAppointment.getPatientIdentifier().getUniqueId(), false,
            bookAppointment.getClinicId(),
            removeTrailingZeros(vistaDate),
            bookAppointment.getPurpose(),
            "",
            bookAppointment.getApptLength(),
            bookAppointment.getApptType(),
            bookAppointment.getBookingNotes(),
            bookAppointment.getLab(),
            bookAppointment.getxRay(),
            bookAppointment.getEkg(),
            bookAppointment.getLvl(),
            vistaDesiredDate);

        BookAppointment bookedAppointment;
        String error = varresponseValidator.validateMdwsObject(lvAppointmentTO);
        if (error != null && error.length() > 0) {
            bookedAppointment = new BookAppointment();
            bookedAppointment.setError(error);
            return bookedAppointment;
        } else {
            MdwsResponseValidator.validateMdwsObject(lvAppointmentTO);
        }
        AppointmentTimeSlotTranslator translator = new AppointmentTimeSlotTranslator();
        return translator.translate(lvAppointmentTO, siteCode);
    }

    public BookedAppointments getPatientBookedAppointments(AppointmentSlotFilter appointmentSlotFilter) {
        logger.debug("Fetch patient booked appointments");
        SchedulingConnection authenticatedConnection = super.getAuthenticatedConnection(appointmentSlotFilter.getSiteCode());
        TaggedAppointmentArray taggedAppointmentArray = authenticatedConnection.getPort().getVeteranAppointmentsBookingNotes(
            appointmentSlotFilter.getSiteCode(),
            appointmentSlotFilter.getPatientIdentifier().getUniqueId(), false,
            DateHelper.convertDateTimeToVistaDateTimeFormat(appointmentSlotFilter.getStartDate()));
        MdwsResponseValidator.validateMdwsObject(taggedAppointmentArray);
        BookedAppointmentTranslator translator = new BookedAppointmentTranslator();
        return translator.translateFromTaggedAppointmentArray(taggedAppointmentArray);
    }

    public CancelAppointment cancelAppointmentForPatient(CancelAppointment cancelAppointmentObject, String siteCode) {
        logger.debug("Cancel appointment for patient");
        String appointmentDate = DateHelper.convertDateTimeToVistaDateTimeFormat(cancelAppointmentObject.getAppointmentTime());
        SchedulingConnection authenticatedConnection = super.getAuthenticatedConnection(siteCode);
        CancelAppointmentTO cancelAppointmentTO = authenticatedConnection.getPort().cancelVeteranAppointment(siteCode,
            cancelAppointmentObject.getPatientIdentifier().getUniqueId(), false, cancelAppointmentObject.getClinicId(),
            removeTrailingZeros(appointmentDate),
            cancelAppointmentObject.getCancelCode(),
            cancelAppointmentObject.getCancelReason(),
            cancelAppointmentObject.getRemarks());
        MdwsResponseValidator.validateMdwsObject(cancelAppointmentTO);
        CancelAppointmentTranslator translator = new CancelAppointmentTranslator();
        CancelAppointment canceledAppointment = translator.translate(cancelAppointmentTO, siteCode);
        canceledAppointment.setPatientIdentifier(cancelAppointmentObject.getPatientIdentifier());
        return canceledAppointment;
    }

    public AppointmentTimeSlots getClinicSchedulingDetail(AppointmentSlotFilter appointmentSlotFilter) {
        logger.debug("Fetch clinic scheduling detail.");
        if (appointmentSlotFilter.getStartDate() == null && appointmentSlotFilter.getEndDate() == null && appointmentSlotFilter.getClinicId() == null) {
            return new AppointmentTimeSlots();
        }
        SchedulingConnection authenticatedConnection = super.getAuthenticatedConnection(appointmentSlotFilter.getSiteCode());
        TaggedHospitalLocationArray hospitalLocationTO = authenticatedConnection.getPort().getVeteranClinicSchedulingDetailsByDateRange(appointmentSlotFilter.getSiteCode(),
            appointmentSlotFilter.getPatientIdentifier().getUniqueId(), false, appointmentSlotFilter.getClinicId(),
            DateHelper.formatDateTimeInVistaFormat(appointmentSlotFilter.getStartDate()),
            DateHelper.formatDateTimeInVistaFormat(appointmentSlotFilter.getEndDate()));
        MdwsResponseValidator.validateMdwsObject(hospitalLocationTO);
        AppointmentTimeSlotTranslator appointmentTimeSlotTranslator = new AppointmentTimeSlotTranslator();
        AppointmentTimeSlots appointmentTimeSlots = appointmentTimeSlotTranslator.translateFromHospitalLocationTO(hospitalLocationTO, appointmentSlotFilter);
        appointmentTimeSlots.setHasUserAccessToClinic(true);
        return appointmentTimeSlots;
    }

    public CancelReasons getCancelReasonList(PatientIdentifier patientIdentifier, String facilityCode) {
        logger.debug("Fetch List of Appointment Cancel Reason");
        SchedulingConnection authenticatedConnection = super.getAuthenticatedConnection(facilityCode);
        TaggedCancellationReasonsArray taggedCancellationReasonArray = authenticatedConnection.getPort().getVeteranCancellationReasons(facilityCode, patientIdentifier.getUniqueId(), false);
        MdwsResponseValidator.validateMdwsObject(taggedCancellationReasonArray);
        CancelReasonTranslator translator = new CancelReasonTranslator();
        return translator.translate(taggedCancellationReasonArray);
    }

    private static String removeTrailingZeros(String dateString) {
        if (NullChecker.isNotNullish(dateString)) {
            if (dateString.indexOf(".") > 0) {
                dateString = dateString.replaceAll("0*$", "").replaceAll("\\.$", "");
            }
        }
        return dateString;
    }

    public ClinicProfile getClinicProfile(String siteCode, String clinicId, PatientIdentifier patientIdentifier) {
        logger.debug("Get Clinic Profile...");

        SchedulingConnection authenticatedConnection = super.getAuthenticatedConnection(siteCode);
        TaggedHospitalLocationArray taggedHospitalLocationArray = authenticatedConnection.getPort().
            getVeteranClinicSchedulingDetailsByDateRange(siteCode,
                patientIdentifier.getUniqueId(), false, clinicId,
                DateHelper.formatDateTimeInVistaFormat(new Date()),
                DateHelper.formatDateTimeInVistaFormat(new Date()));

        MdwsResponseValidator.validateMdwsObject(taggedHospitalLocationArray);

        ClinicProfileTranslator translator = new ClinicProfileTranslator();
        return translator.translate(taggedHospitalLocationArray, siteCode, clinicId);
    }
}